import { Arret } from './Ligne/Arret';
import { Direction } from './Ligne/Direction';
import { Ligne } from './Ligne/Ligne';
import { Passage } from './Ligne/Passage';
/**
 * Support non officiel de l'API du Service de Transport de l'Agglomération Nancéienne (STAN).
 *
 * Ce module permet de récupérer:
 * - les prochains passages
 * - les lignes
 * - les directions
 * - les arrêts
 *
 */
export declare class Stan {
    /**
    * Lister les lignes du réseau de transport
    * @example ```js
    * const { Stan } = require('stan-api')
    *
    * Stan.getLignes().then(lignes => console.log(lignes))
    * ```
    * @returns {Promise<Ligne[]>} Une liste contenant les lignes du réseau
    */
    static getLignes(): Promise<Ligne[]>;
    /**
     * Liste des arrêts d'une ligne
     *
     * NB: Un arret peut être commun à plusieurs lignes.
     * @param {Ligne} ligne Ligne
     * @example ```js
     * const { Stan } = require('stan-api')
     *
     * const ligneT4 = {
     *   id: 2332,
     *   numlignepublic: 'T4',
     *   osmid: 'line:GST:4-97',
     *   libelle: 'Tempo 4 - Laxou CLB <> Houdemont Porte Sud'
     * }
     * Stan.getArrets(ligneT4).then(arrets => console.log(arrets))
     * ```
     * @returns {Promise<Arret[]>} Une liste contenant les arrêts de la ligne
     */
    static getArrets(ligne: Partial<Ligne> & {
        id: number;
        numlignepublic: string;
    }): Promise<Arret[]>;
    /**
     * Lister les prochains passages d'un arrêt avec le temps d'attente estimé.
     * Il n'est pas nécessaire de préciser une ligne, on récupère alors tous les passages des lignes desservants l'arrêt
     * @param {Arret} arret Arrêt
     * @example ```js
     * const { Stan } = require('stan-api')
     *
     * const arret = {
     *   ligne: {
     *     id: 2332,
     *     numlignepublic: 'T4',
     *     osmid: 'line:GST:4-97',
     *     libelle: 'Tempo 4 - Laxou CLB <> Houdemont Porte Sud'
     *   },
     *   libelle: 'Place Stanislas - Dom Calmet',
     *   osmid: 'stop_point:GST:SP:NYPCL1'
     * }
     * Stan.getProchainsPassages(arret).then(p => {
     *   console.log(`Liste des prochains passages de l'arrêt "${arret.libelle}" de la ligne ${arret.ligne.numlignepublic}:
     * ${p.map(passage => `${passage.direction} - Temps d'attente: ${Math.trunc(passage.temps_min/60)}H${passage.temps_min%60}min`).join('\n')}`)
     * })
     * ```
     * ```
     * Exemple sortie:
     * Liste des prochains passages de l'arrêt "Place Stanislas - Dom Calmet" de la ligne T4:
     * Direction Houdemont Porte Sud - Temps d'attente: 0H1min
     * Direction Houdemont Porte Sud - Temps d'attente: 0H8min
     * Direction Laxou Champ-le-Boeuf - Temps d'attente: 0H1min
     * Direction Laxou Champ-le-Boeuf - Temps d'attente: 0H8min
     * ```
     * @returns {Promise<Passage[]>} Les prochains passages d'un arrêt
     */
    static getProchainsPassages(arret: Partial<Arret> & {
        osmid: string;
        ligne?: Partial<Ligne>;
    }): Promise<Passage[]>;
    /**
     * Lister les directions d'une ligne
     * @param {Ligne} ligne Ligne
     * @example ```js
     * const { Stan } = require('stan-api')
     *
     * const ligneT4 = {
     *   id: 2332,
     *   numlignepublic: 'T4',
     *   osmid: 'line:GST:4-97',
     *   libelle: 'Tempo 4 - Laxou CLB <> Houdemont Porte Sud'
     * }
     * Stan.getDirections(ligneT4).then(directions => console.log(directions))
     * ```
     * @returns {Promise<Direction[]>} Une liste contenant les directions d'une ligne
     */
    static getDirections(ligne: Partial<Ligne> & {
        id: number;
        numlignepublic: string;
    }): Promise<Direction[]>;
    /**
     * Lister les arrêts sur le chemin d'une direction d'une ligne
     * @param {Direction} direction Direction d'une ligne
     * @returns {Promise<Arret[]>} Les arrêts de la direction
     */
    static getArretsDirection(direction: Direction): Promise<Arret[]>;
    /**
     * Rechercher un arrêt et obtenir l'identifiant associé (OSMID) nécessaire pour obtenir les prochains passages
     * @param {string} request Le nom d'un arrêt à chercher
     * @returns Une liste contenant les arrêts et les identifiants associés
     */
    static getArretOsmid(request: string): Promise<{
        osmid: string;
        libelle: string;
    }[]>;
    private static getClient;
}
